<?php

/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/**
 * Extended implementation of Excel importer for product related functionalities
 *
 * @author Eranga
 */
class ProductExcelImporter extends ExcelImporter {

    protected $suppliers = [];
    protected $categories = [];
    protected $subcategories = [];
    protected $products = [];
    protected $productCostPriceMetrixes = [];
    protected $productSellPriceMetrixes = [];
    protected $productGroups = [];
    protected $productGroupDetails = [];
    protected $webStoreProductGroups = [];

    //helper functions

    public function letterIncrement($val, $increment = 2) {
        for ($i = 1; $i <= $increment; $i++) {
            $val++;
        }

        return $val;
    }

    protected function generateSuppliers(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            $code = $data['A'];
            //check supplier has already been fetched or created
            if (@array_key_exists($code, $this->suppliers)) {
                continue;
            }

            $criteria = new CDbCriteria;
            $criteria->condition = "supsup_fref =:supref";
            $criteria->params = array(':supref' => $code);
            $supplier = Supplier::model()->find($criteria);

            if (!$supplier) {

                $supplier = new Supplier();
                $supplier->supsup_fref = $code;
                $supplier->supnam = $code;
                $supplier->save(false);
            }

            yield $supplier;
        }
    }

    public function createSuppliers() {

        foreach ($this->generateSuppliers() as $supplier) {
            $this->suppliers[$supplier->supsup_fref] = $supplier;
        }

        return $this;
    }

    public function getSuppliers() {
        return array_filter($this->suppliers);
    }

    protected function generateCategories(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            //make category same as subcategory
            $categoryName = $data['Y'];

            //check subcategory has already been fetched or created
            if (@array_key_exists($categoryName, $this->categories)) {
                continue;
            }

            $criteria = new CDbCriteria;
            $criteria->condition = "name =:catname";
            $criteria->params = array(':catname' => $categoryName);
            $category = ProductCategory::model()->find($criteria);

            if (!$category) {
                $category = new ProductCategory();
            }

            $category->name = $categoryName;
            $category->description = $categoryName;
            $category->createdDate = date('Y-m-d', strtotime('now'));
            $category->createdBy = Yii::app()->user->id;
            $category->save(false);


            yield $category;
        }
    }

    public function createCategories() {
        foreach ($this->generateCategories() as $category) {
            $this->categories[$category->name] = $category;
        }

        return $this;
    }

    public function getCategories() {
        return array_filter($this->categories);
    }

    protected function generateSubcategories(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            //make category same as subcategory
            $categoryName = $data['Y'];
            $category = $this->categories[$categoryName];

            $subcatName = $data['Y'];

            //check subcategory has already been fetched or created
            if (@array_key_exists($subcatName, $this->subcategories)) {
                continue;
            }

            $criteria = new CDbCriteria;
            $criteria->condition = "name =:subcatname";
            $criteria->params = array(':subcatname' => $subcatName);
            $subcategory = ProductSubCategory::model()->find($criteria);

            if (!$subcategory) {
                $subcategory = new ProductSubCategory();
            }

            $subcategory->productCategoryId = $category ? $category->id : null;
            $subcategory->name = $subcatName;
            $subcategory->description = $subcatName;
            $subcategory->createdDate = date('Y-m-d', strtotime('now'));
            $subcategory->createdBy = Yii::app()->user->id;
            $subcategory->save(false);


            yield $subcategory;
        }
    }

    public function createSubcategories() {
        foreach ($this->generateSubcategories() as $subcategory) {
            $this->subcategories[$subcategory->name] = $subcategory;
        }

        return $this;
    }

    public function getSubcategories() {
        return array_filter($this->subcategories);
    }

    protected function generateProducts(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];

            //check product has already been fetched or created
            if (@array_key_exists("{$productCode}", $this->products)) {
                continue;
            }

            $productName = $data['D'];
            $description = $data['E'];

            $image = $data['AC'];

            $supProductCode = $data['BB'];
            $webstoreName = $data['BC'];
            $webstoreUuid = $data['BD'];

            $supplier = $this->suppliers[$data['A']];
            //make category same as subcategory
            $category = $this->categories[$data['Y']];
            $subCategory = $this->subcategories[$data['Y']];


            $criteria = new CDbCriteria;
            $criteria->condition = "pdlpdl =:prdcode";
            $criteria->params = array(':prdcode' => $productCode);
            $product = ProductLine::model()->find($criteria);

            if (!$product) {
                $product = new ProductLine();
            }

            $product->pdlpdl = $productCode;
            $product->pdlprdline = $productName;
            $product->pdldes = $description;
            $product->pdlsupsno = $supplier->supsno;
            $product->categoryId = $category->id;
            $product->subCategoryId = $subCategory->id;
            $product->pdlimage1 = $image;

            $product->sup_product_code = $supProductCode;
            $product->webstore_name = $webstoreName;
            $product->webstore_uuid = $webstoreUuid;

            $product->pdlentddt = date('Y-m-d', strtotime('now'));

            $product->save(false);


            yield $product;
        }
    }

    public function createProducts() {
        foreach ($this->generateProducts() as $product) {
            $this->products["{$product->pdlpdl}"] = $product;
        }

        return $this;
    }

    public function getProducts() {
        return array_filter($this->products);
    }

    public function clearAllPriceMatrixes() {
        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];
            ProductPrice::model()->deleteAll('pdppdlsno =:pdppdlsno', array(':pdppdlsno' => $product->pdlsno));
        }

        return $this;
    }

    public function generateProductCostPriceMetrixes(): Generator {


        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];

            // ------------- price break-ups for cost prices ---------------------

            $q1 = 0;
            $q2 = 0;
            $p = 0;
            $level = 1;
            for ($i = 'H'; $i <= 'V'; $i = $this->letterIncrement($i, 2)) {


                $q1 = $data[$i]; //start from 'H' coloumn
                $q2 = $data[$this->letterIncrement($i, 2)]; //jump to 'J' coloumn
                $p = $data[$this->letterIncrement($i, 1)]; // take price from the 'I' coloumn
                //probably skip the header rows
                if (!is_numeric($q1)) {
                    continue;
                }

                //if price is not avilable then break the flow
                if (!$p) {
                    break;
                }

                $matrixArr = array(
                    'pdlsno' => $product->pdlsno,
                    'priceLevel' => $level,
                    'fromQty' => $q1,
                    'toQty' => $q2,
                    'pdpprice' => $p
                );

                $level++;
                // create the qty breaks and yield
                yield ProductPrice::createPriceMetrix($matrixArr);
            }

            // ------------- End price break-ups for cost prices ---------------------
        }
    }

    public function createProductCostPriceMatrixes() {

        foreach ($this->generateProductCostPriceMetrixes() as $productPriceMetrix) {
            $this->productCostPriceMetrixes[$productPriceMetrix->pdppdlsno][] = $productPriceMetrix;
        }

        return $this;
    }

    public function getProductCostPriceMatrixes() {
        return $this->productCostPriceMetrixes;
    }

    public function generateProductSellPriceMetrixes(): Generator {



        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];


            // ------------- price break-ups for sell prices ---------------------

            $q1 = 0;
            $q2 = 0;
            $p = 0;
            $level = 1;
            for ($i = 'AL'; $i <= 'AZ'; $i = $this->letterIncrement($i, 2)) {


                $q1 = $data[$i]; //start from 'H' coloumn
                $q2 = $data[$this->letterIncrement($i, 2)]; //jump to 'J' coloumn
                $p = $data[$this->letterIncrement($i, 1)]; // take price from the 'I' coloumn
                //probably skip the header rows
                if (!is_numeric($q1)) {
                    continue;
                }

                //if price is not avilable then break the flow
                if (!$p) {
                    break;
                }

                $matrixArr = array(
                    'pdlsno' => $product->pdlsno,
                    'priceLevel' => $level,
                    'fromQty' => $q1,
                    'toQty' => $q2,
                    'pdpprice' => null,
                    'pdpsellprice' => $p
                );

                $level++;
                // create the qty breaks and yield
                yield ProductPrice::createPriceMetrix($matrixArr);
            }

            // ------------- End price break-ups for cost prices ---------------------
        }
    }

    public function createProductSellPriceMatrixes() {

        foreach ($this->generateProductSellPriceMetrixes() as $productPriceMetrix) {
            $this->productSellPriceMetrixes[$productPriceMetrix->pdppdlsno][] = $productPriceMetrix;
        }

        return $this;
    }

    public function getProductSellPriceMatrixes() {
        return $this->productCostPriceMetrixes;
    }

    protected function generateProductGroups(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            $productGroupName = $data['B']; /** new querment to use B column for product group name */
            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];

            if (!$product) {
                continue;
            }

            $productCodeParts = explode('.', $productCode);

            //if no two parts then cannot be create group product
            if (!$productCodeParts || count($productCodeParts) !== 2) {
                continue;
            }

            //take the first part of the name before dot. i.e. "6WWS.02225" ====> group code becomes "6WWS"
            $productGroupCode = $productCodeParts[0];

            //check product has already been fetched or created
            if (@array_key_exists($productGroupCode, $this->productGroups)) {
                continue;
            }

            $productGroupNames = explode('|', $data['D']); // split product name by "|" and take first part. i.e. "JBs Work Sock 3 Pack - Navy | Orange - King" ===> "JBs Work Sock 3 Pack - Navy"

            $groupName = $productGroupCode; // default to group code - this will be used as short descrption
            if ($productGroupName || count($productGroupNames)) {
                $groupName = $productGroupName ? $productGroupName : $productGroupNames[0];
            }


            $image = $data['AC'];

            $criteria = new CDbCriteria;
            $criteria->condition = "code =:groupCode";
            $criteria->params = array(':groupCode' => $productGroupCode);
            $productGroup = ProductGroup::model()->find($criteria);

            if (!$productGroup) {

                $productGroup = new ProductGroup();
            }

            $productGroup->code = $productGroupCode;
            $productGroup->categoryId = $product->categoryId;
            $productGroup->subCategoryId = $product->subCategoryId;
            $productGroup->short_desc = $groupName;
            $productGroup->long_desc = $groupName;
            $productGroup->image = $image;
            $productGroup->created_date = date('Y-m-d', strtotime('now'));
            $productGroup->created_by = Yii::app()->user->id;

            $productGroup->save(false);


            yield $productGroup;
        }
    }

    public function createProductGroups() {
        foreach ($this->generateProductGroups() as $productGroup) {
            $this->productGroups[$productGroup->code] = $productGroup;
        }

        return $this;
    }

    public function getProductGroups() {
        return array_filter($this->productGroups);
    }

    public function allocateProductsToGroups() {
        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];

            //check product has already been fetched or created
            if (!$product) {
                continue;
            }

            $productCodeParts = explode('.', $productCode);

            //if no two parts then cannot be create group product
            if (!$productCodeParts || count($productCodeParts) !== 2) {
                continue;
            }

            //take the first part of the name before dot. i.e. "6WWS.02225" ====> group code becomes "6WWS"
            $productGroupCode = $productCodeParts[0];
            $productGroup = $this->productGroups[$productGroupCode];

            //check product group has already been fetched or created
            if (!$productGroup) {
                continue;
            }

            $criteria = new CDbCriteria;
            $criteria->condition = "product_group_id =:prodGroupId AND pdlsno =:productCode";
            $criteria->params = array(':prodGroupId' => $productGroup->id, ':productCode' => $product->pdlsno);
            $productGroupDetail = ProductGroupDetail::model()->find($criteria);
            //if product group has been created for the same group is and product id then skip
            if ($productGroupDetail) {
                continue;
            }

            $productGroupDetail = new ProductGroupDetail();
            $productGroupDetail->product_group_id = $productGroup->id;
            $productGroupDetail->pdlsno = $product->pdlsno;
            $productGroupDetail->save(false);

            $this->productGroupDetails[$productGroupCode][] = $productGroupDetail;
        }

        return $this;
    }

    public function getProductGroupDetails() {
        return array_filter($this->productGroupDetails);
    }

    public function generateProductGroupsToWebStores(): Generator {
        foreach ($this->extractedExcelData as $key => $data) {

            $productCode = $data['C'];
            $product = $this->products["{$productCode}"];

            //check product has already been fetched or created
            if (!$product) {
                continue;
            }

            $productCodeParts = explode('.', $productCode);

            //if no two parts then cannot be create group product
            if (!$productCodeParts || count($productCodeParts) !== 2) {
                continue;
            }

            //take the first part of the name before dot. i.e. "6WWS.02225" ====> group code becomes "6WWS"
            $productGroupCode = $productCodeParts[0];
            $productGroup = $this->productGroups[$productGroupCode];

            //check product group has already been fetched or created
            if (!$productGroup) {
                continue;
            }

            if (!$product->webstore_uuid) {
                continue;
            }

            //if product has been assinged for multiple web stores
            $webStoresArr = explode('@', $product->webstore_uuid);
            
           

            if (!count($webStoresArr)) {
                continue;
            }

            foreach ($webStoresArr as $storeUuid) {
            $criteria = new CDbCriteria;
            $criteria->condition = "uuid =:uuid";
                $criteria->params = array(':uuid' => trim($storeUuid));
            $webStore = WebStore::model()->find($criteria);

            if (!$webStore) {                    
                continue;
            }

            if (isset($this->webStoreProductGroups[$webStore->id][$productGroup->id])) {
                continue;
            }

            $criteria2 = new CDbCriteria;
            $criteria2->condition = "webstoreId =:webstoreId AND product_group_id =:prodGroupId";
            $criteria2->params = array(':webstoreId' => $webStore->id, ':prodGroupId' => $productGroup->id);
            $webStoreProductGroup = WebstoreProductGroup::model()->find($criteria2);

            if (!$webStoreProductGroup) {
                $webStoreProductGroup = new WebstoreProductGroup();
            }

            $webStoreProductGroup->webstoreId = $webStore->id;
            $webStoreProductGroup->product_group_id = $productGroup->id;
            $webStoreProductGroup->created_date = date('Y-m-d', strtotime('now'));
            $webStoreProductGroup->created_by = Yii::app()->user->id;
                if(!$webStoreProductGroup->save(false)) {
                    $this->logs[] = "Coud not save web store product group {$productGroup->short_desc} for web store id : {$webStore->id}";
                }

            yield $webStoreProductGroup;
        }
    }
    }

    public function allocateProductGroupsToWebStores() {
        foreach ($this->generateProductGroupsToWebStores() as $webStoreProductGroup) {
            $this->webStoreProductGroups[$webStoreProductGroup->webstoreId][$webStoreProductGroup->product_group_id] = $webStoreProductGroup;
        }

        return $this;
    }

    public function getWebStoreProductGroups() {
        return array_filter($this->webStoreProductGroups);
    }

    public function writeLogs() {
        $rowCount = count($this->extractedExcelData);
        $fileName = basename($this->excelFilePath);

        if ($rowCount && $fileName) {
            $this->logs[] = "Data has been extracted by [{$rowCount}] rows from the {$fileName} file.";

            $supplierCount = count($this->suppliers);
            if ($supplierCount) {
                $supplierMarkup = implode(',', array_keys($this->suppliers));
                $supplierCount ? $this->logs[] = "[{$supplierMarkup}] suppliers have been created/updated from the [{$fileName}] file." : _;
            }

            $productGroupCount = count($this->productGroups);
            if ($productGroupCount) {
                $productGroupMarkup = implode(',', array_keys($this->productGroups));
                $supplierCount ? $this->logs[] = "Items have been grouped into the following groups [{$productGroupMarkup}]" : _;
            }

            $webStoreCount = count($this->webStoreProductGroups);
            if ($webStoreCount) {
                $webStoreMarkup = implode(',', array_keys($this->webStoreProductGroups));
                $webStoreMarkup ? $this->logs[] = "Product groups have been allocated to the following web stores [{$webStoreMarkup}]" : _;
            }
        }

        return $this;
    }

    

}
