<?php

/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/**
 * Description of MYOBAuth
 *
 * @author Eranga Perera <rajaera@gmail.com>
 */
class MYOBAuth
{

	const API_BASE_URL = 'https://api.myob.com/accountright/';
	const API_ACCESS_CODE_URL = 'https://secure.myob.com/oauth2/account/authorize';
	const API_ACCESS_TOKEN_URL = 'https://secure.myob.com/oauth2/v1/authorize';
	const API_SCOPE = 'CompanyFile';

	protected $curlCall;
	protected $settings;
	protected $token;
	protected $guide;
	protected $session_token_id_name = 'myob_token_id';
	protected $curl_header; //CurlHeader

	public function __construct(MyobSetting $setting)
	{
		$this->settings = $setting;
	}

	/**
	 * Get access token by access code
	 * @return array access token info array
	 */
	protected function getAccessToken($accessCode)
	{
		$token_params = array(
			'client_id' => trim($this->settings->api_key),
			'client_secret' => trim($this->settings->api_secret),
			'scope' => self::API_SCOPE,
			'code' => $accessCode,
			'redirect_uri' => trim(strtolower($this->settings->redirect_url)), /* all letters of the url should be in lowercase, otherwise it may cause error */
			'grant_type' => 'authorization_code', // authorization_code -> gives you an access token
		);

		$this->curlCall = new CurlCall(new CurlHeaderAccessToken());
		/* return curl response direclty which should have tokens */

		$response = $this->curlCall->curlPost(self::API_ACCESS_TOKEN_URL, http_build_query($token_params));

		if ($this->curlCall->getResponseCode() == 200 && $response) {
			return json_decode($response);
		}

		return null;
	}

	protected function getRefreshToken()
	{
		$token_params = array(
			'client_id' => trim($this->settings->api_key),
			'client_secret' => trim($this->settings->api_secret),
			'refresh_token' => Yii::app()->session['refresh_token'],
			'grant_type' => 'refresh_token', // refresh_token -> refreshes your access token
		);

		$this->curlCall = new CurlCall(new CurlHeaderAccessToken());
		/* return curl response direclty which should have tokens */
		$response = $this->curlCall->curlPost(self::API_ACCESS_TOKEN_URL, http_build_query($token_params));
		if ($this->curlCall->getResponseCode() == 200 && $response) {
			return json_decode($response);
		}

		return null;
	}

	public function refreshAccessToken()
	{

		$refreshToken = $this->getRefreshToken();

		if (!$refreshToken) {
			throw new Exception('MYOB Access Token Refresh Exception!');
		}

		$tokenExpiresIn = time() + $refreshToken->expires_in; /* default time */
		if ($this->settings->access_token_expires_in) {
			$tokenExpiresIn = time() + ((int) $this->settings->access_token_expires_in * 60);
		}
		$this->token = MYOBToken::model()->findByPk(Yii::app()->session['token_id']);



		$this->token->access_token = $refreshToken->access_token;
		$this->token->refresh_token = $refreshToken->refresh_token;

		$this->token->access_token_expires_in = $tokenExpiresIn;
		if ($this->token->save()) {
			Yii::app()->session['token_id'] = $this->token->id;
			Yii::app()->session['access_token'] = $refreshToken->access_token;
			Yii::app()->session['access_token_expires'] = $tokenExpiresIn;
			Yii::app()->session['refresh_token'] = $refreshToken->refresh_token;
			return true;
		}

		return false;
	}

	/**
	 * Register MYOB access tokens and create token object 
	 * @return MYOBToken if successfully registered else false
	 */
	public function registerToken($accessCode = '', $sessionTokenIdName = 'myob_token_id')
	{

		if ($accessCode) {
			$this->session_token_id_name = $sessionTokenIdName;
			$accessToken = $this->getAccessToken($accessCode); /* direct curl response */
			if (!$accessToken) {
				throw new Exception('MYOB Access Token Retrieve Exception!');
			}

			$this->token = new MYOBToken();
			$this->token->access_token = $accessToken->access_token;
			$this->token->refresh_token = $accessToken->refresh_token;
			$tokenExpiresIn = time() + $accessToken->expires_in; /* default time */
			if ($this->settings->access_token_expires_in) {
				$tokenExpiresIn = time() + ((int) $this->settings->access_token_expires_in * 60);
			}

			$this->token->access_token_expires_in = $tokenExpiresIn;
			$this->token->account_owner_uuid = $accessToken->user->uid;
			$this->token->account_owner_username = $accessToken->user->username;
			$this->token->created_at = date('Y-m-d H:i:s');
			$this->token->created_by = Yii::app()->user->id;

			if ($this->token->save()) {

				Yii::app()->session['token_id'] = $this->token->id;
				Yii::app()->session['access_token'] = $accessToken->access_token;
				Yii::app()->session['access_token_expires'] = $tokenExpiresIn;
				Yii::app()->session['refresh_token'] = $accessToken->refresh_token;

				return true;
			}
		}

		return false;
	}

	/**
	 * Retrieve MYOB company accounts and update the guide table
	 * @return curl response company account list
	 */
	protected function retrieveGuides()
	{
		
		$this->curlCall = new CurlCall($this->curl_header);
		/* return curl response direclty which should resposes with guids */
		$response = $this->curlCall->curlGet(self::API_BASE_URL);

		
		if ($this->curlCall->getResponseCode() == 200 && $response) {
			return json_decode($response);
		}

		return null;
	}

	/**
	 * Update local guide table based on the currently retrieve guide (account) list
	 */
	public function setGuide($guideName)
	{
		$guidResponse = $this->retrieveGuides();
		if (!$guidResponse) {
			throw new Exception('MYOB Company Files Retrieve Exception!');
		}

		//clear the giude table
		MYOBGuide::model()->deleteAll("id > 0");
		$setGuide = false;
		foreach ($guidResponse as $companyFile) {


			if (strtolower(trim($guideName)) === strtolower($companyFile->Name)) {
				$guid = new MYOBGuide();
				$guid->uuid = $companyFile->Id;
				$guid->account_name = $companyFile->Name;
				$guid->library_path = $companyFile->LibraryPath;
				$guid->country = $companyFile->Country;
				$guid->launcher_id = $companyFile->LauncherId;
				$guid->serial_number = $companyFile->SerialNumber;
				if ($setGuide = $guid->save()) {

					Yii::app()->session['myob_guid_uuid'] = $guid->uuid;
				} else {
					throw new Exception('MYOB Company Files Not Updated Exception!');
				}
			}
		}

		return $setGuide;
	}

	/**
	 * try to init MYOBToken, and if it's available return true
	 * @return boolean
	 */
	public function checkAccess()
	{
		if (!isset(Yii::app()->session['access_token'])) {
			return false;
		}

		if (!Yii::app()->session['myob_guid_uuid']) {
			$guiduuid = trim($this->settings->myob_guid_uuid);
			if (!$this->guide = MYOBGuide::model()->find("uuid = '{$guiduuid}'")) {
				return false;
			} else {
				Yii::app()->session['myob_guid_uuid'] = $guiduuid;
			}
		} else {
			$guiduuid = trim(Yii::app()->session['myob_guid_uuid']);
			$this->guide = MYOBGuide::model()->find("uuid = '{$guiduuid}'");
		}

		$this->token = MYOBToken::model()->findByPk(Yii::app()->session['token_id']);


		if (Yii::app()->session['access_token_expires'] > time()) {
			return true;
		} else {
			return $this->refreshAccessToken();
		}
	}

	/**
	 * @return MYOBToken
	 */
	public function getToken()
	{
		return $this->token;
	}

	/**
	 * @return MYOBGuide
	 */
	public function getGuide()
	{
		return $this->guide;
	}

	/**
	 * @return MyobSetting
	 */
	public function getSetting()
	{
		return $this->settings;
	}

	/**
	 * retrieve token id from session
	 * @return string token id
	 */
	public function getTokenId()
	{
		return Yii::app()->session[$this->session_token_id_name];
	}

	/**
	 * Sign in url creation
	 * @return string sign in URL
	 */
	public function createSigninURL()
	{
		$encodedUrl = urlencode($this->settings->redirect_url);
		return self::API_ACCESS_CODE_URL . '?client_id=' . $this->settings->api_key . '&redirect_uri=' . $encodedUrl . '&response_type=code&scope=' . self::API_SCOPE;
	}

	public function setCurlHeaderAccount(CurlHeader $curlHeader)
	{
		$this->curl_header = $curlHeader;
	}

}
