<?php

/**
 * Synchronizing logic of record
 *
 * @author Eranga Perera <rajaera@gmail.com>
 */
class MYOBSynchronizer
{

	const SYNCHRONIZE_TYPE_BOTH = 1;
	const SYNCHRONIZE_TYPE_PULL = 2;
	const SYNCHRONIZE_TYPE_PUSH = 3;

	private $sysnhronizeType = 1;
	public $myobReferenceFinder;
	public $myobConnector;
	public $activeRecord;

	public function __construct(\MYOBApiConnector $myobConnector, CActiveRecord $activeRecord = null)
	{
		$this->myobConnector = $myobConnector;
		if ($activeRecord && !$activeRecord instanceof MYOBSynchronizable) {
			throw new Exception('Active record is not implemented from MYOBSynchronizable!');
		}
		$this->activeRecord = $activeRecord;
	}

	/**
	 * Synchronize specific record based on a given active record
	 * @return boolean success true, otherwise false;
	 */
	public function syncronize()
	{

		switch ($this->sysnhronizeType) {
			case self::SYNCHRONIZE_TYPE_BOTH:
				if ($myobResponse = $this->myobReferenceFinder->hasRecord($this->myobConnector, $this->activeRecord)) { /* assign the response object to $myobResponse if record is available in MYOB */
					return $this->syncronizeOnDate($myobResponse); /* decide whether update local from MYOB or vice versa */
				} else {
					/* record is not available in MYOB anyway. so insert the local record into MYOB */
					return $this->postNewIntoMYOB();
				}
				break;
			case self::SYNCHRONIZE_TYPE_PULL:
				if ($myobResponse = $this->myobReferenceFinder->hasRecord($this->myobConnector, $this->activeRecord)) { /* assign the response object to $myobResponse if record is available in MYOB */
					return $this->syncronizeIntoLocal($myobResponse);
				}
				break;
			case self::SYNCHRONIZE_TYPE_PUSH:
				if ($myobResponse = $this->myobReferenceFinder->hasRecord($this->myobConnector, $this->activeRecord)) { /* assign the response object to $myobResponse if record is available in MYOB */
					return $this->syncronizeIntoMYOB($myobResponse);
				} else {
					return $this->postNewIntoMYOB();
				}
				break;
		}
	}

	/**
	 * Synchronize all records both in local and MYOB by fetching records from MYOB
	 * This should create a new local record based on MYOB record if not available and vice versa
	 * @param string $className name of the ActiveRecord class
	 * @param string $requestUrl full api URL
	 */
	public function syncronizeAll($className, $requestUrl)
	{


		$myobResponses = $this->myobConnector->apiGet(null, $requestUrl); /* pass null as first aurgument to use default curl header */


		if ($this->myobConnector->getCurlResponseCode() == 200 && $myobResponses) { /* if response success and not empty array */

			$GLOBALS['syncAllTotalToBeSync'] = $GLOBALS['syncAllTotalToBeSync'] ?: $myobResponses["Count"];
			$GLOBALS['syncAllCount'] = $GLOBALS['syncAllCount'] ?: 0;
			foreach ($myobResponses as $obj) {
				$GLOBALS['syncAllCount'] ++;
				if ($obj instanceof MYOBResponse) {

					switch ($this->sysnhronizeType) {
						case self::SYNCHRONIZE_TYPE_BOTH:
							if ($queryRow = $this->myobReferenceFinder->hasRecordInLocal($obj)) {
								$this->activeRecord = $className::model()->findByPk($queryRow['id']);
								$this->syncronizeOnDate($obj); /* decide whether update local from MYOB or vice versa */
//						
							} else { /* create new record in local */

								$this->activeRecord = null;
								$this->syncronizeIntoLocal($obj, $className);
							}
							break;
						case self::SYNCHRONIZE_TYPE_PULL:
							if ($queryRow = $this->myobReferenceFinder->hasRecordInLocal($obj)) {

								$this->activeRecord = $className::model()->findByPk($queryRow['id']);
								$this->syncronizeIntoLocal($obj); /* update local record */
//						
							} else { /* create new record in local */

								$this->activeRecord = null;
								$this->syncronizeIntoLocal($obj, $className); /* create new local record based on MYOB record */
							}
							break;
					}
					/* out put count stream */
					echo "{$GLOBALS['syncAllCount']}/{$GLOBALS['syncAllTotalToBeSync']}~"; /* "~" output stream seperator */
					//echo str_pad('', 4096) . "\n"; /* if output straming is not working, try by uncommenting this. This will add more characters to output buffer */
					ob_flush();
					flush();
				}
			}



			if (isset($myobResponses["NextPageLink"])) {
				$this->syncronizeAll($className, $myobResponses["NextPageLink"]);
			} else {
				unset($GLOBALS['syncAllTotalToBeSync']);
				unset($GLOBALS['syncAllCount']);
			}
		}
	}

	function microtime_float()
	{
		list($usec, $sec) = explode(" ", microtime());
		return ((float) $usec + (float) $sec);
	}

	protected function syncronizeOnDate(MYOBResponse $myobResponse)
	{
		if (isset($myobResponse->LastModified)) {
			/* record is availabel and need to figure out the latest record */
			/* then need to check  for the latest record! which one is latest, local or MYOB one! */
			$datetimeMyob = new DateTime(date('Y-m-d H:i:s', strtotime($myobResponse->LastModified)));
			$datetimeLocal = new DateTime($this->activeRecord->myobLastModified());
			if ($datetimeMyob != $datetimeLocal) {
				if ($datetimeLocal > $datetimeMyob) {/* local one is the latest! syncronyze the local record into MYOB */
					return $this->syncronizeIntoMYOB($myobResponse);
				} elseif ($datetimeLocal < $datetimeMyob) {/* MYOB one is the latest! update the local record */
					return $this->syncronizeIntoLocal($myobResponse);
				}
			}
		} else {/* noway to decide the latest one, so update the local record */
			return $this->syncronizeIntoLocal($myobResponse);
		}





		return false;
	}

	protected function syncronizeIntoMYOB(MYOBResponse $myobResponse)
	{


		/* before update the record in MYOB we need to set row version of the local record as MYOB record */
		if ($this->activeRecord->updateMYOBReferences($myobResponse)) {/* make sure local data has myob references, otherwise myob will throw duplicate record error */

			$response = $this->myobConnector->apiPost($this->activeRecord->toJsonString());

			if ($this->myobConnector->getCurlResponseCode() == 200) {

				//again we need to get the same record from MYOB because of post request returns empty body
				$response = $this->myobConnector->apiGet(); /* no need to pass header as it's already set to default */
				if ($this->myobConnector->getCurlResponseCode() == 200 && $response[0]) {
					// now we need to update the local record's row version again, because of updating MYOB record wil change the row version of the MYOB record					
					/* Also need update the modified date. this will prevent unnessary updates from MYOB */

					$this->activeRecord->updateByMYOB($response[0]);					
					$this->activeRecord->reportMYOBSuccess($response[0]);
				} else {
					$GLOBALS['MYOB_ERROR_LINE'] = __LINE__;
					$GLOBALS['MYOB_ERROR_CLASS'] = __CLASS__;
					$this->setError($response);
				}

				return true;
			} else {
				$GLOBALS['MYOB_ERROR_LINE'] = __LINE__;
				$GLOBALS['MYOB_ERROR_CLASS'] = __CLASS__;
				$this->setError($response);
				return false;
			}
		} else {
			/* MYOB references of the local record must be updated according to the MYOB data. Make sure local data has already been updated or is updated, then must return true */
			throw new Exception('MYOB References of the Local Record Not Updated Exception!', 304);
		}
		return false;
	}

	protected function syncronizeIntoLocal(MYOBResponse $myobResponse, $className = null)
	{

		if ($this->activeRecord) {
			
			$this->activeRecord->reportMYOBSuccess($myobResponse);
			return $this->activeRecord->updateByMYOB($myobResponse);
		} elseif ($className) {
			return $className::createByMYOB($myobResponse);
		} else {
			return false;
		}
	}

	protected function postNewIntoMYOB()
	{

		$response = $this->myobConnector->apiPost($this->activeRecord->toJsonString());
		if ($this->myobConnector->getCurlResponseCode() == 201) { /* New record has been created on MYOB */
			//again we need to get the posted record from MYOB because of post request dosn't resturn body
			$response = $this->myobConnector->apiGet(); /* no need to pass header as it's already set to default */
			if ($this->myobConnector->getCurlResponseCode() == 200 && $response[0]) {
				// now we need to update the local record's row version again, because of updating MYOB record will change the row version of the MYOB record
				$this->activeRecord->updateByMYOB($response[0]);
				
				
				$this->activeRecord->reportMYOBSuccess($response[0]);
			} else {
				/* something went wrong when getting the uploaded record */
				$GLOBALS['MYOB_ERROR_LINE'] = __LINE__;
				$GLOBALS['MYOB_ERROR_CLASS'] = __CLASS__;
				$this->setError($response);
			}
			return true; /* return true despite other MYOB calling responses, because of new record has been created in MYOB */
		} else {
			$GLOBALS['MYOB_ERROR_LINE'] = __LINE__;
			$GLOBALS['MYOB_ERROR_CLASS'] = __CLASS__;
			$this->setError($response);
			return false;
		}
	}

	public function setMYOBReferenceFinder(MYOBReferenceFinder $myobReferenceFinder)
	{
		$this->myobReferenceFinder = $myobReferenceFinder;
	}

	/**
	 * Set the type of MYOB data synchronizing. by default the syncronizer will try to syncronize MYOB and local system data
	 * you can set it to one way synchronizing
	 * @param integer $type syncronizing type
	 */
	public function setSynchronizeType($type = 1)
	{
		if (in_array($type, array(self::SYNCHRONIZE_TYPE_BOTH, self::SYNCHRONIZE_TYPE_PULL, self::SYNCHRONIZE_TYPE_PUSH))) {
			$this->sysnhronizeType = $type;
		}
	}

	protected function setError($response)
	{
		if (isset($response['Errors'])) {
			$this->activeRecord->reportMYOBError($response[0]);
		}

		unset($GLOBALS['MYOB_ERROR_LINE']);
		unset($GLOBALS['MYOB_ERROR_CLASS']);
	}

}
