<?php

/**
 * This is the model class for table "myob_log".
 *
 * The followings are the available columns in table 'myob_log':
 * @property string $id
 * @property string $table_name
 * @property string $table_reference_id
 * @property integer $log_type_id
 * @property string $log_message
 * @property string $log_class
 * @property integer $log_line
 * @property string $created_at
 * @property integer $created_by
 */
class MYOBLog extends CActiveRecord
{

	const LOG_TYPE_SUCCESS = 1;
	const LOG_TYPE_ERROR = 2;
	const LOG_MESSAGE_CREATED = 'Record has been created by MYOB';
	const LOG_MESSAGE_UPDATE = 'Record has been updated by MYOB';
	const LOG_MESSAGE_REFERENCE_UPDATE = 'MYOB references has been updated by MYOB';
	const LOG_MESSAGE_ERROR = 'MYOB error has been reported!';

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return MYOBLog the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'myob_log';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('log_type_id, created_by, log_line', 'numerical', 'integerOnly' => true),
			array('table_name, log_class', 'length', 'max' => 100),
			array('table_reference_id', 'length', 'max' => 10),
			array('log_message, created_at', 'safe'),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('id, table_name, table_reference_id, log_type_id, log_message, created_at, created_by', 'safe', 'on' => 'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'table_name' => 'Item Name',
			'table_reference_id' => 'Reference',
			'log_type_id' => 'Log Type',
			'log_message' => 'Log Message',
			'created_at' => 'Created At',
			'created_by' => 'Created By',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id, true);
		$criteria->compare('table_name', $this->table_name, true);
		$criteria->compare('table_reference_id', $this->table_reference_id, true);
		$criteria->compare('log_type_id', $this->log_type_id);
		$criteria->compare('log_message', $this->log_message, true);
		$criteria->compare('created_at', $this->created_at, true);
		$criteria->compare('created_by', $this->created_by);

		$pageSize = (int) GlobalVars::get('RecordsViewPerPage');

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
			'pagination' => array('pageSize' => $pageSize),
			'sort' => array(
				'defaultOrder' => 'created_at DESC',
			)
		));

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
		));
	}

	/**
	 * this will write a myob log by referencing table and table id
	 * log messages can be filtered by log type which is a constant of MYOBLog class
	 * also the class and line which writes the log can be provided
	 * @param array $logConfig activeRecord, logType, message keys with values should be provided otherwise exception will be thrown
	 * activeRecord class should have following public functions : tableName() and getId()
	 */
	public static function writeLog(array $logConfig)
	{
		try {

			$row = Yii::app()->db->createCommand()
							->select('id')->from('myob_log')
							->where('table_name=:TBLNAME AND table_reference_id=:TBLREFID', array(':TBLNAME' => $logConfig['table_name'], ':TBLREFID' => $logConfig['table_reference_id']))->queryRow();
			if ($row && $row['id']) {
				return Yii::app()->db->createCommand()
								->update('myob_log', array(
									'log_type_id' => $logConfig['log_type_id'],
									'log_message' => $logConfig['log_message'],
									'log_class' => $logConfig['log_class'],
									'log_line' => $logConfig['log_line'],
									'created_at' => date('Y-m-d H:i:s'),
									'created_by' => Yii::app()->user->id
										), "id={$row['id']}");
			} else {
				return Yii::app()->db->createCommand()
								->insert('myob_log', array(
									'table_name' => $logConfig['table_name'],
									'table_reference_id' => $logConfig['table_reference_id'],
									'log_type_id' => $logConfig['log_type_id'],
									'log_message' => $logConfig['log_message'],
									'log_class' => $logConfig['log_class'],
									'log_line' => $logConfig['log_line'],
									'created_at' => date('Y-m-d H:i:s'),
									'created_by' => Yii::app()->user->id
				));
			}
		} catch (Exception $e) {
			throw new Exception($e->getMessage(), $e->getCode());
		}
	}

	/**
	 * Get log message based on a format
	 * @return string
	 */
	public function getMessage()
	{
		$obj = @unserialize($this->log_message);
		if ($obj && $obj instanceof MYOBResponse) {
			switch ($this->log_type_id) {
				case MYOBLog::LOG_TYPE_ERROR:
					return MYOBResponse::formatError($obj);
				case MYOBLog::LOG_TYPE_SUCCESS:
					return MYOBResponse::formatSuccess($obj);	
				default : return '';
			}
		} else {
			return $this->log_message;
		}
	}

	/**
	 * string representation of the log_type_id
	 * @return string
	 * @throws Exception "Undefined Log Type" if log type not existed
	 */
	public function typeName()
	{
		switch ($this->log_type_id) {
			case self::LOG_TYPE_SUCCESS:
				return 'SUCCESS';
			case self::LOG_TYPE_ERROR:
				return 'ERROR';
			default:
				throw new Exception('Undefined Log Type!');
		}
	}

	/**
	 * Format the reference number according to the needs
	 * @return string
	 */
	public function refFormat()
	{
		switch ($this->table_name) {
			case 'invoices':
				return Invoice::model()->findByPk($this->table_reference_id)->invoiceNum;
			case 'purchase_orders':
				return PurchaseOrders::model()->findByPk($this->table_reference_id)->poNum;
			default :
				return $this->table_reference_id; // return default
		}
	}

}
