<?php

/**
 * This is the model class for table "myob_tax_codes".
 *
 * The followings are the available columns in table 'myob_tax_codes':
 * @property integer $id
 * @property string $uuid
 * @property string $code
 * @property string $description
 * @property string $type
 * @property string $rate
 * @property string $row_version
 * @property integer $myob_guid_uuid MYOB account id ( CompanyFile )
 */
class MyobTaxCode extends CActiveRecord implements MYOBSynchronizable
{

	/**
	 * Provide the logic to be synchronized this record in an API 
	 * @var MYOBSynchronizer concrete implementation of Synchronizer
	 */
	public $synchronizer;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return MyobTaxCode the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'myob_tax_codes';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('code,type,description,rate', 'required'),
			array('id, taxCollectedAccountId, taxPaidAccountId, systemTaxRateId', 'numerical', 'integerOnly' => true),
			array('uuid, myob_guid_uuid', 'length', 'max' => 36),
			array('code', 'length', 'max' => 3),
			array('description', 'length', 'max' => 30),
			array('type, row_version', 'length', 'max' => 255),
			array('rate', 'length', 'max' => 8),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('id, uuid, code, description, type, rate, row_version', 'safe', 'on' => 'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'uuid' => 'Uuid',
			'code' => 'Code',
			'description' => 'Description',
			'type' => 'Type',
			'rate' => 'Rate',
			'row_version' => 'Row Version',
			'taxCollectedAccountId' => 'Tax Collected Account',
			'taxPaidAccountId' => 'Tax Paid Account',
			'systemTaxRateId' =>'System Tax Rate'
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id);
		$criteria->compare('uuid', $this->uuid, true);
		$criteria->compare('code', $this->code, true);
		$criteria->compare('description', $this->description, true);
		$criteria->compare('type', $this->type, true);
		$criteria->compare('rate', $this->rate, true);
		$criteria->compare('row_version', $this->row_version, true);

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
		));
	}

	public function myobLastModified()
	{
		return false;
	}

	/**
	 * @override
	 * Synchronize current record with MYOB
	 * @param MYOBApiConnector $myobConnector api connection
	 * @return boolean success true, otherwise false
	 */
	public function myobSynchronize(\MYOBApiConnector $myobConnector)
	{
		return $this->synchronizer->syncronize(); /* Hide the syncronizing logics in syncronizer object */
	}

	public function setSynchronizer(\MYOBSynchronizer $synchronizer)
	{
		$this->synchronizer = $synchronizer;
	}

	public function toJsonString()
	{
		$taxarr = array(
			"Code" => $this->code,
			"Description" => substr($this->description, 0, 30),
			"Type" => $this->type,
			"Rate" => $this->rate,
		);

		if ($this->taxCollectedAccountId) {
			$accountRawData = Yii::app()->db->createCommand()->select('uuid')->from('myob_accounts')->where('id=:ID', array(':ID' => $this->taxCollectedAccountId))->queryRow();

			if ($accountRawData) {
				$taxarr["TaxCollectedAccount"] = array(
					"UID" => $accountRawData['uuid'],
				);
			} else {
				$taxarr["TaxCollectedAccount"] = null;
			}
		}

		if ($this->taxPaidAccountId) {
			$accountRawData = Yii::app()->db->createCommand()->select('uuid')->from('myob_accounts')->where('id=:ID', array(':ID' => $this->taxPaidAccountId))->queryRow();

			if ($accountRawData) {
				$taxarr["TaxPaidAccount"] = array(
					"UID" => $accountRawData['uuid'],
				);
			} else {
				$taxarr["TaxPaidAccount"] = null;
			}
		}

		if (!empty($this->uuid) && !empty($this->row_version)) {
			$taxarr["UID"] = $this->uuid;
			$taxarr["RowVersion"] = $this->row_version;
		}


		return json_encode($taxarr);
	}

	public function updateByMYOB(\MYOBResponse $response)
	{
		try {
			Yii::app()->db->createCommand()
					->update('myob_tax_codes', array(
						'uuid' => $response->UID,
						'code' => $response->Code,
						'description' => $response->Description,
						'type' => $response->Type,
						'rate' => $response->Rate,
						'myob_guid_uuid' => Yii::app()->session['myob_guid'],
						'row_version' => $response->RowVersion,
						'json_response' => null,
							), "id=$this->id" /* where condition */
			);
		} catch (Exception $e) {
			throw new Exception($e->getMessage(), $e->getCode());
		}

		return true;
	}

	public function updateMYOBReferences(\MYOBResponse $response)
	{
		try {
			Yii::app()->db->createCommand()
					->update('myob_tax_codes', array(
						'uuid' => $response->UID,
						'myob_guid_uuid' => Yii::app()->session['myob_guid'],
						'row_version' => $response->RowVersion,
						'json_response' => null,
							), "id=$this->id" /* where condition */
			);
		} catch (Exception $e) {
			throw new Exception($e->getMessage(), $e->getCode());
		}
		
		return true;
	}

	public static function createByMYOB(\MYOBResponse $response)
	{
		try {
			Yii::app()->db->createCommand()
							->insert('myob_tax_codes', array(
								'uuid' => $response->UID,
								'code' => $response->Code,
								'description' => $response->Description,
								'type' => $response->Type,
								'rate' => $response->Rate,								
								'row_version' => $response->RowVersion,
								'myob_guid_uuid' => Yii::app()->session['myob_guid']
			));
		} catch (Exception $e) {
			throw new Exception($e->getMessage(), $e->getCode());
		}
		
		return true;
	}

	public function reportMYOBError(\MYOBResponse $response)
	{
		Yii::app()->db->createCommand()
				->update('myob_tax_codes', array(
					'json_response' => serialize($response),
						), "id=$this->id" /* where condition */
		);		

	}

	public function getMYOBError()
	{
		$obj = @unserialize($this->json_response); //In case the passed string is not unserializeable, supress  E_NOTICE 
		return $obj ? MYOBResponse::formatError($obj) : '';
	}

	public function reportMYOBSuccess(\MYOBResponse $response)
	{
		
	}

}
