<?php
/**
* Rights assignment controller class file.
*
* @author Christoffer Niska <cniska@live.com>
* @copyright Copyright &copy; 2010 Christoffer Niska
* @since 0.9.1
*/
class AssignmentController extends RController
{
	/**
	* @property RAuthorizer
	*/
	private $_authorizer;

	/**
	* Initializes the controller.
	*/
	public function init()
	{
		$this->_authorizer = $this->module->getAuthorizer();
		$this->layout = $this->module->layout;
		$this->defaultAction = 'view';

		// Register the scripts
		$this->module->registerScripts();
	}

	/**
	* @return array action filters
	*/
	public function filters()
	{
		//return array('accessControl');
		return array('rights');
	}

	/**
	* Specifies the access control rules.
	* This method is used by the 'accessControl' filter.
	* @return array access control rules
	*/
	public function accessRules()
	{
		return array(
			array('allow', // Allow superusers to access Rights
				'actions'=>array(
					'view',
					'user',
					'revoke',
					'addusers'
				),
				'users'=>$this->_authorizer->getSuperusers(),
			),
			array('deny', // Deny all users
				'users'=>array('*'),
			),
		);
	}

	/**
	* Displays an overview of the users and their assignments.
	*/
	public function actionView()
	{
		// Create a data provider for listing the users
		$dataProvider = new RAssignmentDataProvider(array(
			'pagination'=>array(
				'pageSize'=>50,
			),
		));

		// Render the view
		$this->render('view', array(
			'dataProvider'=>$dataProvider,
		));
	}
	

	/**
	* Displays the authorization assignments for an user.
	*/
	public function actionUser()
	{
		// Only super user having id of 0 can change his own settings
		// Others are not allowed
		if (!$_GET['id'] && 
			$this->module->superuserId != Yii::app()->user->id)
		{
			Yii::app()->user->setFlash($this->module->flashErrorKey,
				Rights::t('core', 'Operation not allowed.')
			);
			$this->redirect(array('assignment/view'));
		}
		
		// Only users with Administrator role will be able to change
		// others role
		$permissions = $this->_authorizer->getAuthItems(CAuthItem::TYPE_ROLE, $_GET['id']);
		if (!Yii::app()->user->getIsAdminRole() && 
			$this->_authorizer->hasRole(Rights::module()->adminRoleName, $permissions))
		{
			Yii::app()->user->setFlash($this->module->flashErrorKey,
				Rights::t('core', 'Operation not allowed.')
			);
			
			$this->redirect(array('assignment/view'));
		}
				
		// Create the user model and attach the required behavior
		$userClass = $this->module->userClass;
		$model = CActiveRecord::model($userClass)->findByPk($_GET['id']);
		$this->_authorizer->attachUserBehavior($model);

		$assignedItems = $this->_authorizer->getAuthItems(null, $model->getId());
		
		$assignments = array_keys($assignedItems);
		$assignments[] = $this->module->superuserName;

		if (!Yii::app()->user->getIsAdminRole())
		{			
			// This workaround is to show his own role(s) only if user is not an
			// Administrative role
			
			$all_items = $this->_authorizer->getAuthItems(CAuthItem::TYPE_ROLE);
			$all_items_keys = array_keys($all_items);
			
			$logged_user_items = $this->_authorizer->getAuthItems(CAuthItem::TYPE_ROLE, Yii::app()->user->id);
			$logged_user_keys = array_keys($logged_user_items);
			
			$diff = array_diff($all_items_keys, $logged_user_keys);
			
			foreach ($diff as $diff_item)
			{
				$assignments[] = $diff_item;
			}
		}

		// Make sure we have items to be selected
		$assignSelectOptions = Rights::getAuthItemSelectOptions(CAuthItem::TYPE_ROLE, $assignments);
		
		if( $assignSelectOptions!==array() )
		{
			$formModel = new AssignmentForm();

		    // Form is submitted and data is valid, redirect the user
		    if( isset($_POST['AssignmentForm'])===true )
			{
				$formModel->attributes = $_POST['AssignmentForm'];
				if( $formModel->validate()===true )
				{
					// Update and redirect
					$this->_authorizer->authManager->assign($formModel->itemname, $model->getId());
					$item = $this->_authorizer->authManager->getAuthItem($formModel->itemname);
					$item = $this->_authorizer->attachAuthItemBehavior($item);

					Yii::app()->user->setFlash($this->module->flashSuccessKey,
						Rights::t('core', 'Permission :name assigned.', array(':name'=>$item->getNameText()))
					);
					
					//Rights::writeLog('Permission assigned.User id:' . $model->getId() . ',Itemname:' . $item->getNameText());

					$this->redirect(array('assignment/user', 'id'=>$model->getId()));
				}
			}
		}
		// No items available
		else
		{
		 	$formModel = null;
		}

		// Create a data provider for listing the assignments
		$dataProvider = new RAuthItemDataProvider('assignments', array(
			'userId'=>$model->getId(),
		));

		// Render the view
		$this->render('user', array(
			'model'=>$model,
			'dataProvider'=>$dataProvider,
			'formModel'=>$formModel,
			'assignSelectOptions'=>$assignSelectOptions,
		));
	}

	/**
	* Revokes an assignment from an user.
	*/
	public function actionRevoke()
	{
		// We only allow deletion via POST request
		if( Yii::app()->request->isPostRequest===true )
		{
			$itemName = $this->getItemName();
			
			if (!Yii::app()->user->checkAccess($itemName) && !Yii::app()->user->getIsAdminRole())
			{
				Yii::app()->user->setFlash($this->module->flashErrorKey,
					Rights::t('core', 'Operatin not allowed.')
				);
				
				$this->redirect(array('assignment/user', 'id'=>$_GET['id']));
			}
			
			// Revoke the item from the user and load it
			$this->_authorizer->authManager->revoke($itemName, $_GET['id']);
			$item = $this->_authorizer->authManager->getAuthItem($itemName);
			$item = $this->_authorizer->attachAuthItemBehavior($item);

			// Set flash message for revoking the item
			Yii::app()->user->setFlash($this->module->flashSuccessKey,
				Rights::t('core', 'Permission :name revoked.', array(':name'=>$item->getNameText()))
			);

			//Rights::writeLog('Permission revoked.User id:' . $_GET['id'] . ',Itemname:' . $item->getNameText());			

			// if AJAX request, we should not redirect the browser
			if( isset($_POST['ajax'])===false )
				$this->redirect(array('assignment/user', 'id'=>$_GET['id']));
		}
		else
		{
			throw new CHttpException(400, Rights::t('core', 'Invalid request. Please do not repeat this request again.'));
		}
	}
	
	/**
	* @return string the item name or null if not set.
	*/
	public function getItemName()
	{
		return isset($_GET['name'])===true ? urldecode($_GET['name']) : null;
	}
}
